import sys

sys.path.append("../../oemsws")

from samples import _log
from oemsws.api import API
from oemsws.persistence import TokenPersistence
from samples.keycloak.config import (
    API_ENDPOINT,
    TOKEN_ENDPOINT,
    CLIENT_ID,
    USER,
    GROUP,
)
import json

_log.setup()


class JsonTokenPersistence(TokenPersistence):
    """Token persistence strategy that stores refresh token strings in a JSON file.
    Stores a token for each client_id as key-value pairs. See custom_token_persistence.py for usage.
    The implementation below is intentionally simple for the explanation purpose.
    """

    def __init__(self, filename, client_id):
        self.filename = filename
        self.client_id = client_id

    def save(self, token):
        obj = self._read_json()
        obj[self.client_id] = token
        self._write_json(obj)

    def load(self):
        obj = self._read_json()
        return obj[self.client_id]

    def _read_json(self):
        with open(self.filename, "r") as f:
            return json.load(f)

    def _write_json(self, obj):
        with open(self.filename, "w") as f:
            json.dump(obj, f, indent=2)


def main():
    # Create an API instance and open a connection.
    with API.create_persistence(
        API_ENDPOINT,
        TOKEN_ENDPOINT,
        CLIENT_ID,
        GROUP,
        USER,
        token_persistence=JsonTokenPersistence(
            "samples/custom_token_persistence.json", CLIENT_ID
        ),
    ) as api:

        def on_update(update):
            print(f"Order update received: {update}")

        api.order_service.subscribe(
            {"subscriptionId": "test order subscription", "symbolType": "tora"},
            on_update,
        )
        input("Hit enter to stop...\n")

    # The end of `with` statement closes the connection
    print("Connection closed.")


if __name__ == "__main__":
    main()
