import csv
import sys
import uuid
from datetime import datetime
from logging import getLogger

from oemsws.api import API
from oemsws.persistence import ClientCredentialsTokenPersistence
from samples import _log
from samples.client_credentials.config import (
    API_ENDPOINT,
    TOKEN_ENDPOINT,
    CLIENT_ID,
    USER,
    GROUP,
    CLIENT_SECRET,
)

sys.path.append("../../../oemsws")
_log.setup()

logger = getLogger(__name__)


def main():
    orders = []
    # read orders from CSV
    try:
        with open(
            "samples/client_credentials/orders/data/tora_orders.csv", mode="r"
        ) as csv_file:
            csv_reader = csv.DictReader(csv_file)
            for row in csv_reader:
                if row.get("clientOrderId", None) is None:
                    row["clientOrderId"] = str(uuid.uuid4())
                orders.append({"order": row})
    except FileNotFoundError:
        logger.error("CSV file not found...")
        return
    except csv.Error as e:
        logger.error(f"CSV error occurred: {e}")
        return

    if len(orders) == 0:
        logger.info("Nothing to register...")
        return

    output_file_name = "batch_order_ids_{date:%Y-%m-%d_%H-%M-%S}.csv".format(
        date=datetime.now()
    )
    output_file_path = f"samples/client_credentials/orders/data/{output_file_name}"

    def on_close():
        logger.info("API connection closed.")

    def on_error(error: Exception):
        if error is not None:
            logger.error(error)
        logger.error("Error in ws connection.")

    # Create an API instance and open a connection.
    with (
        API.create_persistence(
            API_ENDPOINT,
            TOKEN_ENDPOINT,
            CLIENT_ID,
            GROUP,
            USER,
            on_close=on_close,
            on_error=on_error,
            token_persistence=ClientCredentialsTokenPersistence(
                "oemsws_access_token.txt"
            ),
            auth_type="ping",
            client_secret=CLIENT_SECRET,
        ) as api,
        open(output_file_path, mode="w+", newline="") as out_f,
    ):
        batch_result = api.order_service.batch(orders, action="register")
        writer = csv.writer(out_f)

        # Write the header
        writer.writerow(["orderId", "statusCode", "statusMessage"])

        # Write the order IDs
        for order in batch_result:
            try:
                writer.writerow([order["orderId"], "0", "OK"])
            except TypeError:
                writer.writerow(["", order.code, order.message])

        logger.info(f"Registered order with orderIds: {batch_result}")
        input("Hit enter to stop...\n")

    # The end of `with` statement closes the connection
    print("Connection closed.")


if __name__ == "__main__":
    main()
