import datetime
import sys
import uuid

from oemsws.api import API
from oemsws.exceptions import OemsException
from oemsws.persistence import ClientCredentialsTokenPersistence
from samples import _log
from samples.client_credentials.config import (
    API_ENDPOINT,
    TOKEN_ENDPOINT,
    CLIENT_ID,
    USER,
    GROUP,
    CLIENT_SECRET,
)

sys.path.append("../../../oemsws")
_log.setup()


def main():
    # Create an API instance and open a connection.
    with API.create_persistence(
        API_ENDPOINT,
        TOKEN_ENDPOINT,
        CLIENT_ID,
        GROUP,
        USER,
        token_persistence=ClientCredentialsTokenPersistence("oemsws_access_token.txt"),
        auth_type="ping",
        client_secret=CLIENT_SECRET,
    ) as api:

        # This handler approves all violated limits
        def on_compliance_update(update):
            data = update.data
            print(f"Compliance update: {data}")
            for violation in data.get("complianceViolations", []):
                decision_time = datetime.datetime.utcnow().strftime(
                    "%Y-%m-%dT%H:%M:%S.%fZ"
                )
                # Finds violated orders with no decision (=pending approval)
                orders_to_be_approved = (
                    order
                    for order in violation["complianceViolatingOrders"]
                    if "decision" not in order
                )
                for order in orders_to_be_approved:
                    try:
                        api.compliance_service.send_approval_response(
                            {
                                "complianceViolation": {
                                    "complianceViolationId": violation[
                                        "complianceViolationId"
                                    ],
                                    "complianceViolatingOrders": [
                                        {
                                            "orderId": order["orderId"],
                                            "decision": {
                                                "decisionMaker": USER,
                                                "decisionTime": decision_time,
                                                "status": "approved",
                                            },
                                        }
                                    ],
                                }
                            },
                            timeout=10.0,
                        )
                    except OemsException as exc:
                        print(exc)

        api.compliance_service.subscribe(
            {
                "subscriptionId": str(uuid.uuid4()),
                "symbolType": "reuters",
            },
            on_compliance_update,
        )

        input("Hit enter to stop...\n")

    # The end of `with` statement closes the connection
    print("Connection closed.")


if __name__ == "__main__":
    main()
