from functools import wraps
from logging import getLogger
from typing import List, Tuple

from pydantic import ValidationError
from oemsws.exceptions import ValidationException, OemsException

logger = getLogger(__name__)


def validate_pair_order(schema, algo_schema):
    """
    Decorator. Validates order data.
    """

    def vd(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                # extract order and algo params
                order_data_first = args[1]
                order_data_second = args[2]
                algo_params = args[3]

                # validate against model
                schema(**order_data_first)
                schema(**order_data_second)
                algo_schema(**algo_params)

                return func(*args, **kwargs)
            except ValidationError as e:
                raise ValidationException(
                    "validation failed.", [field["loc"] for field in e.errors()]
                )

        return wrapper

    return vd


def validate_native_atdl_pair_order(schema):
    """
    Decorator. Validates order data.
    """

    def vd(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                # extract orders
                order_data_first = args[1]
                order_data_second = args[2]

                # validate against model
                schema(**order_data_first)
                schema(**order_data_second)

                return func(*args, **kwargs)
            except ValidationError as e:
                raise ValidationException(
                    "validation failed.", [field["loc"] for field in e.errors()]
                )

        return wrapper

    return vd


def validate(schema):
    """
    Decorator. Validates user input data.
    """

    def vd(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                # extract user_data
                user_data = args[1]
                # validate against model
                schema(**user_data)
                return func(*args, **kwargs)
            except ValidationError as e:
                raise ValidationException(
                    str(e), [field["loc"] for field in e.errors()]
                )

        return wrapper

    return vd


def validate_batch(schema):
    """
    Decorator for batch.
    Validates user input data.
    Skips invalid data.
    """

    def vd(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                # extract user_data
                user_data_list: List = args[1]
                if not isinstance(user_data_list, list):
                    raise TypeError(
                        "Invalid batch data. Should be a list of order object."
                    )
                for user_data in user_data_list:
                    # validate against model
                    schema(**user_data)
                return func(*args, **kwargs)
            except ValidationError as e:
                logger.exception("Invalid order data: %s" % user_data)
                raise ValidationException(
                    str(e), [field["loc"] for field in e.errors()]
                )
            except Exception as e:
                raise OemsException(str(e))

        return wrapper

    return vd
