from typing import Dict, Callable

from oemsws.models import ServerMessage
from oemsws.constants import TRADING_CAPABILITIES, TIMEOUT
from oemsws.schema import Subscribe
from oemsws.session import Session, Subscription
from oemsws.validator import validate


class TradingCapabilityService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=Subscribe)
    def subscribe(
        self,
        params: Dict,
        on_update: Callable[[ServerMessage], None],
        *,
        timeout=TIMEOUT
    ) -> Subscription:
        """Subscribe the available trading capabilities.

        :param params: a dict object of subscribe action parameters
            described in Tora API documentation.
        :param on_update: a handler called on update. The first argument of the
            handler is a Response object. The handler is
            executed on a new thread. Unhandled exceptions in the handler are
            ignored.
        :param timeout: timeout period in seconds.
        :return: Subscription object with an unsubscribe method that
            unsubscribes the created subscription.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        return self._session.subscribe(TRADING_CAPABILITIES, params, on_update, timeout)
