from typing import Dict, Callable

from oemsws.constants import TIMEOUT, CANCEL, REQUEST, QUOTES
from oemsws.models import ServerMessage, Request
from oemsws.schema import QuotesCancel, QuotesRequest, Subscribe
from oemsws.session import Session, Subscription
from oemsws.validator import validate


class QuotesService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=Subscribe)
    def subscribe(
        self,
        params: Dict,
        on_update: Callable[[ServerMessage], None],
        *,
        timeout=TIMEOUT,
    ) -> Subscription:
        """Subscribe quotes.

        :param params: a dict object of subscribe action parameters
            described in Tora API documentation.
        :param on_update: a handler called on update. The first argument of the
            handler is a Response object. The handler is
            executed on a new thread. Unhandled exceptions in the handler are
            ignored.
        :param timeout: timeout period in seconds.
        :return: Subscription object with an unsubscribe method that
            unsubscribes the created subscription.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        return self._session.subscribe(QUOTES, params, on_update, timeout)

    @validate(schema=QuotesRequest)
    def request(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Request quotes.

        :param params: a dict object of request action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: a dict object with order information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        response = self._send_request(REQUEST, params, timeout)
        return response.data or {}

    @validate(schema=QuotesCancel)
    def cancel(self, params: Dict, *, timeout=TIMEOUT):
        """Cancel quotes.

        :param params: a dict object of cancel action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(CANCEL, params, timeout)

    def _send_request(self, action: str, params: Dict, timeout) -> ServerMessage:
        request = Request(QUOTES, action)
        request.params = params
        return self._session.send(request, timeout)
