from typing import Dict

from oemsws.models import Request
from oemsws.constants import (
    PRODUCT,
    GENERIC_SEARCH,
    MARKET_PROPERTIES,
    PRODUCT_DETAILS,
    TIMEOUT,
)
from oemsws.schema import ProductSearch, ProductMarketProperties, ProductDetails
from oemsws.session import Session
from oemsws.validator import validate


class ProductService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=ProductSearch)
    def search(self, params: Dict, *, timeout=TIMEOUT) -> dict:
        """Search for product information.

        :param params: a dict object of search action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: dict with product information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        request = Request(PRODUCT, GENERIC_SEARCH)
        request.params = params
        response = self._session.send(request, timeout)
        return response.data or {}

    @validate(schema=ProductMarketProperties)
    def market_properties(self, params: Dict, *, timeout=TIMEOUT) -> dict:
        """Get market properties.

        :param params: a dict object of market properties action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: dict with market properties.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        request = Request(PRODUCT, MARKET_PROPERTIES)
        request.params = params
        response = self._session.send(request, timeout)
        return response.data or {}

    @validate(schema=ProductDetails)
    def product_details(self, params: Dict, *, timeout=TIMEOUT) -> dict:
        """Get product details.

        :param params: a dict object of product details parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: dict with product information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        request = Request(PRODUCT, PRODUCT_DETAILS)
        request.params = params
        response = self._session.send(request, timeout)
        return response.data or {}
