from logging import getLogger
from typing import Dict

from oemsws.constants import (
    TIMEOUT,
    REGISTER,
    REGISTER_AND_SEND,
    PAIRS,
    SEND,
    AMEND,
    CANCEL,
    PAIRUP,
)
from oemsws.models import Request, ServerMessage
from oemsws.schema import PairRegisterParams, PairCancel, PairSend, PairUp
from oemsws.session import Session
from oemsws.validator import validate

logger = getLogger(__name__)


class PairService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=PairRegisterParams)
    def register(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Registers a pair.

        :param params: a dict object of pairs register action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: a dict object with order information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        response = self._send_request(REGISTER, params, timeout)
        return response.data or {}

    @validate(schema=PairSend)
    def send(self, params: Dict, *, timeout=TIMEOUT):
        """Sends a pair.

        :param params: a dict object of pairs send action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(SEND, params, timeout)

    @validate(schema=PairRegisterParams)
    def register_and_send(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Registers and sends a pair.

        :param params: a dict object of pairs registerAndSend action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: a dict object with order information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        response = self._send_request(REGISTER_AND_SEND, params, timeout)
        return response.data or {}

    def amend(self, params: Dict, *, timeout=TIMEOUT):
        """Amends a pair.

        :param params: a dict object of pairs amend action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(AMEND, params, timeout)

    @validate(schema=PairCancel)
    def cancel(self, params: Dict, *, timeout=TIMEOUT):
        """Cancels a pair.

        :param params: a dict object of pairs cancel action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(CANCEL, params, timeout)

    def _send_request(self, action: str, params: Dict, timeout) -> ServerMessage:
        request = Request(PAIRS, action)
        request.params = params
        return self._session.send(request, timeout)

    @validate(schema=PairUp)
    def pair_up(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Handles the pairUp action.

        :param params: a dict object of pairUp action parameters.
        :param timeout: timeout period in seconds.
        :return: a dict object with order information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        response = self._send_request(PAIRUP, params, timeout)
        return response.data or {}
