from typing import Dict, Callable

from oemsws.constants import COMPLIANCES, TIMEOUT, APPROVAL_DECISION, APPROVAL_RESPONSE
from oemsws.models import Request, ServerMessage
from oemsws.schema import Subscribe
from oemsws.session import Session, Subscription
from oemsws.validator import validate


class ComplianceService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=Subscribe)
    def subscribe(
        self,
        params: Dict,
        on_update: Callable[[ServerMessage], None],
        *,
        timeout=TIMEOUT
    ) -> Subscription:
        """Subscribe compliance notifications.

        :param params: a dict object of subscribe action parameters
            described in Tora API documentation.
        :param on_update: a handler called on update. The first argument of the
            handler is a Response object. The handler is
            executed on a new thread. Unhandled exceptions in the handler are
            ignored.
        :param timeout: timeout period in seconds.
        :return: Subscription object with an unsubscribe method that
            unsubscribes the created subscription.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        return self._session.subscribe(COMPLIANCES, params, on_update, timeout)

    def send_approval_decision(self, params: Dict, *, timeout=TIMEOUT):
        """Sends an approval decision message.

        :param params: a dict object of "approval decision" action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(APPROVAL_DECISION, params, timeout)

    def send_approval_response(self, params: Dict, *, timeout=TIMEOUT):
        """Sends an approval response message.

        :param params: a dict object of "approval response" action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        self._send_request(APPROVAL_RESPONSE, params, timeout)

    def _send_request(self, action: str, params: Dict, timeout):
        request = Request(COMPLIANCES, action)
        request.params = params
        return self._session.send(request, timeout)
