from concurrent import futures
from typing import Callable, Dict, List
from oemsws.constants import BORROW, BORROW_CREATE, TIMEOUT
from oemsws.exceptions import OemsException
from oemsws.models import Request, ServerMessage
from oemsws.schema import BorrowSubscribe, BorrowCreate
from oemsws.session import Session, Subscription
from oemsws.validator import validate, validate_batch


class BorrowService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=BorrowSubscribe)
    def subscribe(
        self,
        params: Dict,
        on_update: Callable[[ServerMessage], None],
        *,
        timeout=TIMEOUT,
    ) -> Subscription:
        """Subscribe borrow requests.

        :param params: a dict object of subscribe action parameters
            described in Tora API documentation.
        :param on_update: a handler called on update. The first argument of the
            handler is a Response object. The handler is
            executed on a new thread. Unhandled exceptions in the handler are
            ignored.
        :param timeout: timeout period in seconds.
        :return: Subscription object with an unsubscribe method that
            unsubscribes the created subscription.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        return self._session.subscribe(BORROW, params, on_update, timeout)

    @validate(schema=BorrowCreate)
    def create(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Create a borrow request.

        :param params: a dict object of register action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: a dict object that contains the response data of the request.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        response = self._send_request(BORROW_CREATE, params, timeout)
        return response.data or {}

    def batch(self, params: List[Dict], *, action=None, timeout=TIMEOUT) -> List:
        """Perform action on a batch of borrow orders.

        :param params: a list of dict object of based on action,
            described in Tora API documentation.
        :param action: action user want to perform on batch
        :param timeout: timeout period in seconds.
        :return: a list of dict object with borrow order information, also include failure.
        """
        action_to_function_map = {
            "create": self.create_batch,
        }
        func: Callable = action_to_function_map.get(action)
        if func is None:
            raise OemsException(
                f"Action is required. Possible actions are {list(action_to_function_map.keys())}"
            )
        return func(params, timeout=timeout)

    @validate_batch(schema=BorrowCreate)
    def create_batch(self, params: List, *, timeout=TIMEOUT) -> List:
        """Create a batch of borrow order.

        :param params: a list of dict object of borrow action parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: a list with borrow order information.
        """

        response = self._send_batch_request(BORROW_CREATE, params, timeout)
        return response

    def _send_batch_request(self, action: str, params: List, timeout) -> List:
        request_batch = []
        for p in params:
            request = Request(BORROW, action)
            request.params = p
            request_batch.append(request)
        return self._session.send_batch(request_batch, timeout)

    def _send_request(self, action: str, params: Dict, timeout) -> ServerMessage:
        request = Request(BORROW, action)
        request.params = params
        return self._session.send(request, timeout)
