from typing import Dict

from oemsws.constants import TIMEOUT, ALGO, ALGO_DETAILS, ALGO_STRATEGIES
from oemsws.models import Request
from oemsws.schema import AlgoStrategies, AlgoDetails
from oemsws.session import Session
from oemsws.validator import validate


class AlgoDefinitionService:
    def __init__(self, session: Session):
        self._session = session

    @validate(schema=AlgoStrategies)
    def algo_strategies(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Get algo strategies.

        :param params: a dict object of algo strategies parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: dict with algo strategies information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        request = Request(ALGO, ALGO_STRATEGIES)
        request.params = params
        response = self._session.send(request, timeout)
        return response.data or {}

    @validate(schema=AlgoDetails)
    def algo_details(self, params: Dict, *, timeout=TIMEOUT) -> Dict:
        """Get algo strategy details.

        :param params: a dict object of algo details parameters
            described in Tora API documentation.
        :param timeout: timeout period in seconds.
        :return: dict with algo strategy detail information.
        :raise Timeout: if a response is not received within the timeout.
        :raise ResponseError: if the received response has error status.
        :raise WebSocketConnectionError: if websocket fails to send the message.
        """
        request = Request(ALGO, ALGO_DETAILS)
        request.params = params
        response = self._session.send(request, timeout)
        return response.data or {}
