import uuid
from logging import getLogger
from typing import Dict, Optional

import simplejson as json

from oemsws.constants import (
    MESSAGE_ID,
    SERVICE,
    ACTION,
    PARAMS,
    EVENT,
    MESSAGE_REFERENCE_ID,
    STATUS,
    DATA,
    CODE,
    MESSAGE,
    OPEN_ID_CONNECT,
    AUTH,
    USER,
    GROUP,
    ID_TOKEN,
    AUTH_METHOD,
    ACCESS_TOKEN,
)
from oemsws.exceptions import ResponseError
from oemsws.__version__ import __version__

logger = getLogger(__name__)


class Request:
    """Request message from API clients (= this library) to the API server."""

    def __init__(self, service: str, action: str):
        self.message_id: str = str(uuid.uuid4())
        self.service: str = service
        self.action: str = action
        self.params: Dict[str, Optional[str]] = dict()

    def to_json(self) -> str:
        payload = {
            MESSAGE_ID: self.message_id,
            SERVICE: self.service,
            ACTION: self.action,
            PARAMS: self.params,
        }
        return json.dumps(payload, use_decimal=True)

    def __repr__(self):
        type_name = type(self).__name__
        return (
            f"<{type_name} message_id={self.message_id}, "
            f"service={self.service}, "
            f"action={self.action}, "
            f"params={self.params}>"
        )


class ServerMessage:
    """Incoming message from the API server.

    The message can be a response to a request or an update for
    a subscription.

    """

    def __init__(self, message_id: str, service: str, event: str):
        self.message_id: str = message_id
        self.message_ref_id: Optional[str] = None
        self.service: str = service
        self.event: str = event
        self.params: Optional[Dict] = None
        self.data: Optional[Dict] = None
        self.status: Optional[Dict] = None

    @classmethod
    def from_json(cls, json_string: str):
        """Converts a string to ServerMessage.

        :param json_string: json format string.
        :return: ServerMessage object
        :raise: OemsException if the status code is non-zero.
        """
        obj = json.loads(json_string, use_decimal=True)
        response = cls(obj[MESSAGE_ID], obj[SERVICE], obj[EVENT])
        response.message_ref_id = obj.get(MESSAGE_REFERENCE_ID, None)
        response.params = obj.get(PARAMS, None)
        response.data = obj.get(DATA, None)
        response.status = obj.get(STATUS, None)
        return response

    def raise_if_error(self):
        """Raise an exception if the message status is non-zero.

        :raise: ResponseError if status code is not zero.
        """
        if self.status and self.status.get(CODE, 0) != 0:
            raise ResponseError(self, self.status[CODE], self.status.get(MESSAGE, ""))

    def __repr__(self):
        type_name = type(self).__name__
        return (
            f"<{type_name} message_id={self.message_id}, "
            f"message_ref_id={self.message_ref_id}, "
            f"service={self.service}, "
            f"event={self.event}, "
            f"params={self.params}, "
            f"data={self.data}, "
            f"status={self.status}>"
        )


class AuthRequest(Request):
    """Represents an authentication request."""

    def __init__(
        self,
        id_token: str,
        group: Optional[str] = None,
        user: Optional[str] = None,
        auth_method: str = OPEN_ID_CONNECT,
    ):
        super().__init__(AUTH, AUTH)
        self.params = {
            AUTH_METHOD: auth_method,
            ID_TOKEN: id_token,
        }
        if group:
            self.params[GROUP] = group
        if user:
            self.params[USER] = user
        if __version__:
            self.params["version"] = __version__


class ClientCredentialsAuthRequest(Request):
    """Represents an authentication request."""

    def __init__(
        self,
        access_token: str,
        group: Optional[str] = None,
        user: Optional[str] = None,
        auth_method: str = OPEN_ID_CONNECT,
    ):
        super().__init__(AUTH, AUTH)
        self.params = {
            AUTH_METHOD: auth_method,
            ACCESS_TOKEN: access_token,
        }
        if group:
            self.params[GROUP] = group
        if user:
            self.params[USER] = user
        if __version__:
            self.params["version"] = __version__
