from typing import Optional


class OemsException(Exception):
    """Root exception for the library.

    All exception types in the library must inherit this type.
    """

    def __init__(self, message: Optional[str] = None):
        self.message: Optional[str] = message

    def __repr__(self):
        type_name = type(self).__name__
        return f"<{type_name} message={self.message}>"

    def __str__(self):
        return self.__repr__()


class Timeout(OemsException):
    """Represents a timeout error."""

    pass


class ResponseError(OemsException):
    """Represents an error received from a websocket server."""

    def __init__(self, response, code: int, message: str):
        self.response = response
        self.code = code
        self.message = message

    def __repr__(self):
        type_name = type(self).__name__
        return (
            f"<{type_name} response={self.response}, code={self.code}, "
            f"message={self.message}>"
        )


class WebSocketConnectionError(OemsException):
    """Represents a connection error."""

    pass


class ValidationException(OemsException):
    """Represent a validation error."""

    def __init__(self, message: str, fields: list):
        self.message = message
        self.fields = fields

    def __repr__(self):
        type_name = type(self).__name__
        return f"<{type_name} message={self.message} params={self.fields}>"

    def __str__(self):
        return self.__repr__()
