from logging import getLogger
from typing import Dict, Optional, Callable

from oemsws.constants import TIMEOUT
from oemsws.models import ServerMessage
from oemsws.openid import TokenStore
from oemsws.persistence import (
    InMemoryTokenPersistence,
    DefaultTokenPersistence,
    TokenPersistence,
)
from oemsws.services.algo_definition import AlgoDefinitionService
from oemsws.services.analytics import AnalyticsService
from oemsws.services.borrow import BorrowService
from oemsws.services.compliance import ComplianceService
from oemsws.services.execution import ExecutionService
from oemsws.services.order import OrderService
from oemsws.services.pair import PairService
from oemsws.services.position import PositionService
from oemsws.services.product import ProductService
from oemsws.services.quotes import QuotesService
from oemsws.services.trading_capability import TradingCapabilityService
from oemsws.session import Session
from oemsws.__version__ import __project_name__, __version__

logger = getLogger(__name__)


class API:
    """Represents a low level API interface.

    `oemsws.API` class provides a low level interface and the following
    functionality:

    - maintain websocket connection
    - refresh tokens with OpenID Connect server and re-authenticate with API
    server

    All method names in `oemsws.API` correspond to **action** in OEMS websocket
    API, except for unsubscribe method.
    """

    @classmethod
    def create(
        cls,
        api_endpoint: str,
        token_endpoint: str,
        client_id: str,
        refresh_token: str,
        group: Optional[str] = None,
        user: Optional[str] = None,
        sslopt: Optional[Dict] = None,
        on_closing: Optional[Callable[[ServerMessage], None]] = None,
        on_close: Optional[Callable[[], None]] = None,
        on_error: Optional[Callable[[Exception], None]] = None,
        on_ping: Optional[Callable[[str], None]] = None,
    ):
        """Creates an API object.
        :param api_endpoint: a websocket endpoint URL. e.g. wss://example.com
        :param token_endpoint: a token endpoint of OpenID Connect.
        :param client_id: a client ID of OpenID Connect.
        :param refresh_token: a refresh token of OpenID Connect.
        :param group: a Compass group name.
        :param user: an user name.
        :param sslopt: SSL options passed to websocket-client.
        :param on_closing: a handler called on closing. The first argument of the
            handler is the server message. The handler is executed on a new thread.
            Unhandled exceptions in the handler are ignored.
        :param on_close: a handler called on close. The handler is executed on a new
            thread. Unhandled exceptions in the handler are ignored.
        :param on_error: a handler called on error. The first argument of the
            handler is the exception. The handler is executed on a new thread. Unhandled
            exceptions in the handler are ignored.
        :param on_ping: a handler called on ping. The first argument of the handler
            is the server send time in epoch ms. The handler is executed on a new
            thread. Unhandled exceptions in the handler are ignored.
        To disable ssl cert verification, specify {"cert_reqs": ssl.CERT_NONE}.
        :raise OemsException: raises when token refresh failed with an
        unexpected error.
        """
        token_persistence: TokenPersistence = InMemoryTokenPersistence(refresh_token)
        return cls.create_persistence(
            api_endpoint,
            token_endpoint,
            client_id,
            group,
            user,
            sslopt,
            on_closing,
            on_close,
            on_error,
            token_persistence,
            on_ping,
        )

    @classmethod
    def create_persistence(
        cls,
        api_endpoint: str,
        token_endpoint: str,
        client_id: str,
        group: Optional[str] = None,
        user: Optional[str] = None,
        sslopt: Optional[Dict] = None,
        on_closing: Optional[Callable[[ServerMessage], None]] = None,
        on_close: Optional[Callable[[], None]] = None,
        on_error: Optional[Callable[[Exception], None]] = None,
        token_persistence: TokenPersistence = DefaultTokenPersistence(),
        on_ping: Optional[Callable[[str], None]] = None,
        auth_type: Optional[str] = "keycloak",
        client_secret: Optional[str] = None,
    ):
        """Creates an API object.
        :param api_endpoint: a websocket endpoint URL. e.g. wss://example.com
        :param token_endpoint: a token endpoint of OpenID Connect.
        :param client_id: a client ID of OpenID Connect.
        :param group: a Compass group name.
        :param user: an user name.
        :param sslopt: SSL options passed to websocket-client.
        :param on_closing: a handler called on closing. The first argument of the
            handler is the server message. The handler is executed on a new thread.
            Unhandled exceptions in the handler are ignored.
        :param on_close: a handler called on close. The handler is executed on a new
            thread. Unhandled exceptions in the handler are ignored.
        :param on_error: a handler called on error. The first argument of the
            handler is the exception. The handler is executed on a new thread. Unhandled
            exceptions in the handler are ignored.
        :param token_persistence: a token persistence object. By default, it saves the last
            refresh token in a file. You can use a custom token persistence object by inheriting
            TokenPersistance abstract class.
        :param on_ping: a handler called on ping. The first argument of the handler
            is the server send time in epoch ms. The handler is executed on a new
            thread. Unhandled exceptions in the handler are ignored.
        :param auth_type: Auth type to use. Default is keycloak.
        :param client_secret: a client secret of OpenID Connect.
        To disable ssl cert verification, specify {"cert_reqs": ssl.CERT_NONE}.
        :raise OemsException: raises when token refresh failed with an
        unexpected error.
        """
        logger.info("Creating API connection using %s-%s", __project_name__, __version__)
        token_store: TokenStore = TokenStore(
            token_endpoint.strip(),
            client_id.strip(),
            token_persistence,
            auth_type=auth_type,
            client_secret=(
                client_secret.strip() if client_secret is not None else client_secret
            ),
        )
        session: Session = Session(
            token_store,
            api_endpoint.strip(),
            group.strip(),
            user.strip(),
            sslopt,
            on_closing=on_closing,
            on_close=on_close,
            on_error=on_error,
            on_ping=on_ping,
        )
        return cls(session)

    def __init__(self, session: Session):
        """Initializes an API object. Use *create* class method instead.
        :param session: a Session object.
        """
        self._session = session
        self.order_service = OrderService(session)
        self.position_service = PositionService(session)
        self.execution_service = ExecutionService(session)
        self.analytics_service = AnalyticsService(session)
        self.trading_capability_service = TradingCapabilityService(session)
        self.compliance_service = ComplianceService(session)
        self.product_service = ProductService(session)
        self.borrow_service = BorrowService(session)
        self.algo_definition_service = AlgoDefinitionService(session)
        self.quotes_service = QuotesService(session)
        self.pair_service = PairService(session)

    def __enter__(self, timeout: float = TIMEOUT):
        self.open(timeout)
        return self

    def __exit__(self, ex_type, ex_value, traceback):
        self.close()

    def open(self, timeout: float = TIMEOUT):
        """Opens an API connection.

        This method opens an API connection. This should not be called more
        than once. If you need multiple connections, create another API object.

        :param timeout: Timeout period in seconds.
        :raise Timeout: if no authentication response is received
            for the timeout period.
        :raise ResponseError: if authentication fails.
        """
        self._session.open(timeout)

    def close(self):
        """Closes an API connection.
        :raise WebSocketConnectionError: if websocket was closed unexpectedly.
        """
        self._session.close()
