# Python wrapper for OEMS API

This is a Python wrapper library of [OEMS API](https://server-api.docs.tora.com/).

This library does:
- maintain websocket connection
- expose actions/queries in simple interfaces 
- refresh OpenID tokens 

Here is a simple example that subscribes orders and sends a DMA order.

```python
from oemsws.api import API

API_ENDPOINT = "wss://path/to/api"
TOKEN_ENDPOINT = "https://path/to/protocol/openid-connect/token"
CLIENT_ID = ""


def main():
    # Ask the user to provide credentials.
    refresh_token = input("refresh token: ")
    user = input("user: ")
    group = input("group: ")
    
    def on_closing(message):
        print(f"Connection is closing: {message}")

    def on_close():
        print(f"Connection closed")

    def on_error(error):
        print(f"Error occurred: {error}")

    # Create an API instance and open a connection.
    with API.create_persistence(API_ENDPOINT, TOKEN_ENDPOINT, CLIENT_ID, group, 
                    user, on_closing, on_close, on_error) as api:
        # subscribe orders
        def on_update(update):
            print(f"Order update received: {update}")

        api.order_service.subscribe({
            "subscriptionId": "test order subscription",
            "symbolType": "tora",
        }, on_update)

        # send a DMA order
        api.order_service.register_and_send({
            "order": {
                "clientOrderId": "client order id",
                "symbol": "CS.NULL.JAPAN.T.JPY.7203",
                "symbolType": "tora",
                "quantity": "100",
                "side": "BUY",
                "limitPrice": "0.0",
                "brokerAccount": "abc",
                "broker": "ms"
            }
        })
        
        input("Hit enter to stop...\n")

    # The end of `with` statement closes the connection    
    print("Connection closed.")
```
For client credentials auth flow, please check `samples/register_worked_client_credentials_auth.py`

# Examples

The following examples are available in samples directory.

- Order Service
    - subscribe_orders.py: subscribe orders.
    - send_dma.py: send a DMA order.
    - register_amend_cancel.py: perform various order actions.
    - parent_child.py: create a child order under a parent order.
- Execution Service
    - executions.py: subscribe executions.
- Position Service
    - positions.py: subscribe positions.
- Analytics Service
    - analytics_order.py: subscribe analytic data for orders.
    - analytics_symbol.py: subscribe analytic data for a symbol.
- Trading Capability Service
    - tradingcapabilities.py: subscribe trading capabilities.
- Compliance Service
    - compliance.py: subscribe compliance violations and approve all.
- Product Service
    - product.py: subscribe product information.
- Borrow Service
    - borrow_subscribe.py: subscribe borrow requests.
    - borrow_create.py: create a borrow request.
- Algo Definition Service
    - algo_definition.py: get information about the definitions of algo strategies.
    
To execute an example, 

1. replace parameters in `./samples/config.py`.
2. create `./oemsws_refresh_token.txt` and paste a refresh token string.
3. run `python samples/example_name.py`.

# Installation

This library can be installed as a Python package, e.g. `pip install .\oemsws`, or if provided as pip package `pip install oemsws-x.x.x.zip`

Python 3.7 and newer is supported.

# Usage

## API Interfaces

`oemsws.API` class provides API interfaces and the following functionality:

- maintain websocket connection
- refresh tokens with OpenID Connect server and re-authenticate with API server
 
All method names in `oemsws.API` correspond to **action** in OEMS websocket API.

For example: 

|OEMS websocket API|This library|
|---|---|
|service="orders", action="subscribe"|API.order_service.subscribe|
|service="orders", action="send"|API.order_service.send|
|service="positions", action="subscribe"|API.position_service.subscribe|

One exception is `unsubscribe` action. `subscribe` method returns `Subscription` object that has `unsubscribe` method 
to the created subscription so that users don't need to remember subscription ID strings.

See the docstring for details.

## Token persistence customization

By default, DefaultTokenPersistence stores the last refresh token in a flat file in the current directory. This behavior should cover most of use cases, but you may want to customize it to support the following cases.

- Store a refresh token for each client_id in a JSON file.
- Store refresh tokens in an encrypted format.
- Store refresh tokens in a database
- Create multiple API instances using a pool of pregenerated refresh tokens.

You can cover them by passing a custom TokenPersistence instance to the `API.create` method. See `samples/custom_token_persistence.py` as an example.

# Changelog

## 1.5.2

### Changed

- Add SDK version in client and server log

## 1.5.1

### Changed

- Add validations for DMA and worked orders
- Allow decimals for order quantity field
- Add coverage of pair-up and sample
- Remove SSL_OPT from samples

## 1.5.0

### Changed

- Add FOREX in ALLOWED_SYMBOL_TYPES list
- Add worked flow validation
- Add short order validation
- Make version optional in algo fields for Pair service registration

## 1.4.0

### Changed

- Trim spaces in user defined config parameters
- Add borrow snapshot samples
- Fix sample path reference

## 1.3.0

### Changed

- Allow NORMAL MARKET orders

## 1.2.0

### Changed

- Add order flow validator
- Update reference application for both keycloak and credentials

## 1.1.0

### Changed

- Handle server unsubscribe event
- Add reference application for keycloak
- Segregate samples for pingId flow and keycloak flow

## 1.0.0

### Changed

- Upgrade Python version to 3.10
- Upgrade dependencies

## 0.8.0

### Changed

- Add support for pair service
- Refactor samples per-service basis

## 0.7.0

### Changed

- Add manualExecution action in orders service

## 0.6.2

### Changed

- Add packages in setuptools

## 0.6.1

### Changed

- Return amend, send, cancel batch response
- Add samples for register, amend, cancel batch

## 0.6.0

### Changed

- Add client credential auth flow
- Add support for native ATDL pairs
- Add support for RFQ service

## 0.5.1

### Changed

- Add workATDLDescription back in order in Pair registration flow
- Update idtoken_gen.py - add option to create token file

## 0.5.0

### Changed

- Updated batch implementation for order and borrow services
- raise ValidationException if any of the batch data is invalid

## 0.4.0

### Changed

- use session.send instead of session.send_async in compliance service
- updated sample

## 0.3.9

### Changed

- added batch actions for order and borrow service
- updated validator for compliance service

## 0.3.8

### Changed

- updated grouping id for register_pair to ensure proper flow downstream 

## 0.3.7

### Changed

- added base validators for all services

## 0.3.6

### Changed

- updated samples
- added validator for conversion of pair order parameters
- added validator for clientOrderId leg uniqueness of pair orders
- added validator and default for primary leg of pair orders
- updated pair order parameters mapping

## 0.3.5

### Changed

- added implementation for pair order registration

## 0.3.4

### Changed

- added implementation for algo definitions endpoint

## 0.3.3

### Changed

- added openid rate limit error handling

## 0.3.2

### Changed

- added openid response error handling and logging
- added refresh token validation

## 0.3.1

### Changed

- added on_ping handler to `API.create`

## 0.3.0

### Changed

- BREAKING CHANGE: moved `API.create` to `API.create_persistence` method. `API.create` now follows <0.2.0 refresh token behavior.

## 0.2.2

### Added

- Added borrow service

## 0.2.1

### Changed 

- Removed version constraints for `requests` and `simplejson` in setup.py


## 0.2.0

### Changed

- BREAKING CHANGE: Removed the `refresh_token` parameter and added TokenPersistence parameter in the `API.create` method. Users need to create `./oemsws_refresh_token.txt` that contains a refresh token if the default TokenPersistence is used.

# Development Setup

Run `pip install -r requirements.txt`.

# Build Distributable

Run the following command:

```bash
python tools/build_dist.py
```

This command generates `dist/oemsws.zip`. It contains pip installable package, and samples.

# Testing

Run `python tools/test-runner.py` in venv.

An HTML coverage report will be generated in htmlcov directory.

# Other Tools

## idtoken_gen.py

Generates OpenID token interactively. 
1. Replace the parameters at the top of the file to specify a token endpoint and a client ID.
2. Run `python tools/idtoken_gen.py`
To run idtoken_gen.py in windows pycharm, Run - Edit Configurations - Execution - 'Select' Emulate terminal in output console. 

## autopep8.py

Formats python code.

## mypy_runner.py

Runs mypy.